<?php
/**
 * Template Name: Contact
 * Description: Secure contact + Sentra messaging (auth gated).
 */
get_header();

$mc_phone = defined('MOORES_PHONE') ? MOORES_PHONE : '';
$quote_url = '';
if (function_exists('moores_sentra_config')) {
	$cfg = moores_sentra_config();
	if (is_array($cfg)) {
		$quote_url = isset($cfg['quote_url']) ? trim((string) $cfg['quote_url']) : '';
		if (!$quote_url) $quote_url = isset($cfg['portal_url']) ? trim((string) $cfg['portal_url']) : '';
	}
}
if (!$quote_url && function_exists('sentrasystems_config')) {
	$cfg = sentrasystems_config();
	if (is_array($cfg) && !empty($cfg['quote_url'])) {
		$quote_url = trim((string) $cfg['quote_url']);
	}
	if (!$quote_url && is_array($cfg) && !empty($cfg['portal_url'])) {
		$quote_url = trim((string) $cfg['portal_url']);
	}
}
$services = function_exists('moores_sentra_get_services') ? moores_sentra_get_services(60) : [];
if (is_array($services)) {
	$services = array_values(array_filter($services, function($service) {
		if (!is_array($service)) return false;
		if (isset($service['is_active']) && !$service['is_active']) return false;
		return !empty($service['title']);
	}));
	usort($services, function($a, $b) {
		$sa = isset($a['sort']) ? (int) $a['sort'] : 0;
		$sb = isset($b['sort']) ? (int) $b['sort'] : 0;
		if ($sa === $sb) return strcasecmp((string) ($a['title'] ?? ''), (string) ($b['title'] ?? ''));
		return $sa <=> $sb;
	});
}

$ajax_url = function_exists('admin_url') ? admin_url('admin-ajax.php') : '';
$nonce = function_exists('wp_create_nonce') ? wp_create_nonce('moores_auth') : '';
$quote_href = $quote_url ? $quote_url : '#';
$quote_disabled = $quote_url ? '' : 'is-disabled';
$mc_phone_raw = $mc_phone ? preg_replace('/[^0-9+]/', '', $mc_phone) : '';
?>

<section class="section" id="contact" data-contact-page data-ajax-url="<?php echo esc_url($ajax_url); ?>" data-nonce="<?php echo esc_attr($nonce); ?>" data-call-number="<?php echo esc_attr($mc_phone_raw); ?>">
	<div class="container">
		<div class="panel edge pad mc-quote-panel">
			<div class="mc-quote-header">
				<h1>Contact & Messaging</h1>
				<p class="lead">
					Start a secure conversation with Moore’s CustomZ. We’ll gather a few details so the team can follow up with accurate answers and timelines.
				</p>
			</div>

			<div class="mc-quote-disclosure">
				<strong>AI assistance notice:</strong>
				<span>AI only helps with the first set of general questions and information gathering. After that, your conversation is marked for human assistance before any pricing or decisions.</span>
			</div>

			<div class="mc-quote-gate" data-quote-gate>
				<p class="lead">Please sign in to access secure messaging.</p>
				<div class="mc-quote-actions">
					<button class="btn btn-primary" type="button" data-auth-open>Sign in to continue</button>
				</div>
			</div>

			<div class="mc-quote-content" data-quote-content hidden data-quote-link="<?php echo esc_url($quote_href); ?>">
				<p class="mc-quote-user">Signed in as <strong data-quote-user>Client</strong></p>
				<form class="mc-quote-form" data-quote-form>
					<div class="mc-quote-grid">
						<label class="mc-field">
							<span>Service needed</span>
							<select name="service" required>
								<option value="">Select a service</option>
								<?php if (!empty($services)) : ?>
									<?php foreach ($services as $service) : ?>
										<option value="<?php echo esc_attr($service['title']); ?>"><?php echo esc_html($service['title']); ?></option>
									<?php endforeach; ?>
								<?php endif; ?>
								<option value="Other">Other / Not sure</option>
							</select>
						</label>

						<label class="mc-field">
							<span>Project type</span>
							<select name="vehicle_type" required>
								<option value="">Select one</option>
								<option value="car">Car / SUV</option>
								<option value="truck">Truck</option>
								<option value="motorcycle">Motorcycle</option>
								<option value="bike">Bike</option>
								<option value="other">Other item</option>
								<option value="not_vehicle">Not vehicle related</option>
							</select>
						</label>

						<div class="mc-field-group" data-vehicle-fields hidden>
							<label class="mc-field">
								<span>Make</span>
								<input type="text" name="make" placeholder="Example: Ford">
							</label>
							<label class="mc-field">
								<span>Model</span>
								<input type="text" name="model" placeholder="Example: F-150">
							</label>
							<label class="mc-field">
								<span>Year</span>
								<input type="text" name="year" inputmode="numeric" placeholder="2020">
							</label>
						</div>

						<label class="mc-field mc-field-full" data-other-item hidden>
							<span>Item details</span>
							<input type="text" name="item_details" placeholder="Describe the item or project (size, material, etc.)">
						</label>

						<label class="mc-field">
							<span>Preferred timeframe</span>
							<select name="timeframe">
								<option value="">Select timeframe</option>
								<option value="asap">ASAP (based on availability)</option>
								<option value="2_weeks">Within 2 weeks</option>
								<option value="1_month">Within 1 month</option>
								<option value="flexible">Flexible</option>
								<option value="specific">Specific date</option>
							</select>
						</label>

						<label class="mc-field" data-timeframe-date hidden>
							<span>Preferred date (not guaranteed)</span>
							<input type="date" name="timeframe_date">
						</label>

						<label class="mc-field">
							<span>Budget range</span>
							<input type="text" name="budget" list="mc-budget-options" placeholder="Under $1,000, $1,000-$3,000, etc.">
							<datalist id="mc-budget-options">
								<option value="Under $1,000"></option>
								<option value="$1,000 – $3,000"></option>
								<option value="$3,000 – $6,000"></option>
								<option value="$6,000+"></option>
								<option value="Not sure yet"></option>
							</datalist>
						</label>

						<label class="mc-field mc-field-full">
							<span>Scope of work</span>
							<textarea name="scope" rows="3" placeholder="Describe the finish, style, or outcome you want."></textarea>
						</label>

						<label class="mc-field mc-field-full">
							<span>Anything else that helps?</span>
							<textarea name="notes" rows="2" placeholder="Deadlines, references, constraints, or must-haves."></textarea>
						</label>
					</div>

					<div class="mc-quote-actions">
						<button class="btn btn-primary" type="submit" data-quote-submit>Continue to chat</button>
						<button class="btn btn-outline" type="button" data-quote-reset>Reset</button>
					</div>
					<div class="mc-quote-status" data-quote-status hidden></div>
				</form>

			</div>
		</div>
	</div>
</section>

<script>
(() => {
	const host = document.querySelector('[data-contact-page]');
	if (!host) return;
	const gate = host.querySelector('[data-quote-gate]');
	const content = host.querySelector('[data-quote-content]');
	const userLabel = host.querySelector('[data-quote-user]');
	const closers = document.querySelectorAll('[data-auth-close]');
	const form = host.querySelector('[data-quote-form]');
	const resetBtn = host.querySelector('[data-quote-reset]');
	const submitBtn = host.querySelector('[data-quote-submit]');
	const submitLabel = submitBtn ? submitBtn.textContent : 'Continue to chat';
	const status = host.querySelector('[data-quote-status]');
	const vehicleType = host.querySelector('[name="vehicle_type"]');
	const vehicleFields = host.querySelector('[data-vehicle-fields]');
	const otherField = host.querySelector('[data-other-item]');
	const timeframeSelect = host.querySelector('[name="timeframe"]');
	const timeframeDate = host.querySelector('[data-timeframe-date]');
	const callNumber = host.dataset.callNumber || '';
	const callHref = callNumber ? `tel:${callNumber}` : '';
	const ajaxUrl = host.dataset.ajaxUrl || '';
	const nonce = host.dataset.nonce || '';
	let messagingAvailable = null;
	let pingPromise = null;

	const getStoredPacket = () => {
		try {
			const raw = localStorage.getItem('sentra_auth_packet');
			if (!raw) return null;
			return JSON.parse(raw);
		} catch (e) {
			return null;
		}
	};

	const isPacketValid = (packet) => {
		if (!packet || !packet.expires_at) return false;
		const now = Math.floor(Date.now() / 1000);
		return packet.expires_at > now;
	};

	const markDismissed = () => {
		try { sessionStorage.setItem('mc_contact_auth_dismissed', '1'); } catch (e) {}
	};

	closers.forEach(btn => btn.addEventListener('click', markDismissed));

	const openAuthModal = () => {
		const opener = document.querySelector('[data-auth-open]');
		if (opener) opener.click();
	};

	const applyState = (packet, autoOpen = false) => {
		const isValid = isPacketValid(packet);
		if (gate) gate.hidden = isValid;
		if (content) content.hidden = !isValid;
		if (userLabel) {
			userLabel.textContent = isValid ? (packet.email || packet.username || 'Client') : 'Client';
		}
		if (!isValid) {
			if (status) status.hidden = true;
		}
		if (isValid && messagingAvailable === null) {
			pingMessaging();
		}
		if (!isValid && autoOpen) {
			try {
				const dismissed = sessionStorage.getItem('mc_contact_auth_dismissed') === '1';
				if (!dismissed) setTimeout(openAuthModal, 80);
			} catch (e) {
				setTimeout(openAuthModal, 80);
			}
		}
	};

	applyState(getStoredPacket(), true);

	window.addEventListener('sentra-auth-updated', (event) => {
		applyState(event?.detail?.packet || null, false);
	});

	window.addEventListener('focus', () => {
		if (document.hidden) return;
		applyState(getStoredPacket(), false);
	});

	const updateVehicleFields = () => {
		if (!vehicleType) return;
		const value = (vehicleType.value || '').toLowerCase();
		const needsVehicle = ['car', 'truck', 'motorcycle', 'bike'].includes(value);
		const needsOther = value === 'other' || value === 'not_vehicle';
		if (vehicleFields) vehicleFields.hidden = !needsVehicle;
		if (otherField) otherField.hidden = !needsOther;
	};

	const updateTimeframe = () => {
		if (!timeframeSelect || !timeframeDate) return;
		const needsDate = timeframeSelect.value === 'specific';
		timeframeDate.hidden = !needsDate;
		if (!needsDate) {
			const input = timeframeDate.querySelector('input');
			if (input) input.value = '';
		}
	};

	if (vehicleType) vehicleType.addEventListener('change', updateVehicleFields);
	if (timeframeSelect) timeframeSelect.addEventListener('change', updateTimeframe);
	updateVehicleFields();
	updateTimeframe();

	const buildSummary = () => {
		if (!form) return '';
		const data = new FormData(form);
		const get = (key) => (data.get(key) || '').toString().trim();
		const lines = [];
		const service = get('service');
		if (service) lines.push(`Service: ${service}`);
		const type = get('vehicle_type');
		if (type) {
			const labels = {
				car: 'Car / SUV',
				truck: 'Truck',
				motorcycle: 'Motorcycle',
				bike: 'Bike',
				other: 'Other item',
				not_vehicle: 'Not vehicle related'
			};
			lines.push(`Project type: ${labels[type] || type}`);
		}
		if (['car', 'truck', 'motorcycle', 'bike'].includes(type)) {
			const make = get('make');
			const model = get('model');
			const year = get('year');
			const parts = [year, make, model].filter(Boolean);
			if (parts.length) lines.push(`Vehicle: ${parts.join(' ')}`);
		}
		if (type === 'other' || type === 'not_vehicle') {
			const item = get('item_details');
			if (item) lines.push(`Item details: ${item}`);
		}
		const timeframe = get('timeframe');
		if (timeframe) {
			const tfLabels = {
				asap: 'ASAP (based on availability)',
				'2_weeks': 'Within 2 weeks',
				'1_month': 'Within 1 month',
				flexible: 'Flexible',
				specific: 'Specific date'
			};
			let tfText = tfLabels[timeframe] || timeframe;
			const date = get('timeframe_date');
			if (date) tfText += ` - ${date}`;
			lines.push(`Timeframe: ${tfText}`);
		}
		const budget = get('budget');
		if (budget) {
			lines.push(`Budget: ${budget}`);
		}
		const scope = get('scope');
		if (scope) lines.push(`Scope: ${scope}`);
		const notes = get('notes');
		if (notes) lines.push(`Notes: ${notes}`);
		return lines.join('\n');
	};

	const proxyRequest = async (resource, method, payload = null, query = null) => {
		if (!ajaxUrl) throw new Error('Proxy unavailable.');
		const body = new FormData();
		body.set('action', 'moores_sentra_proxy');
		body.set('resource', resource);
		body.set('method', method);
		if (nonce) body.set('nonce', nonce);
		if (query && typeof query === 'object') {
			body.set('query', JSON.stringify(query));
		}
		const packet = getStoredPacket();
		if (packet?.access_token) {
			body.set('token', packet.access_token);
		}
		if (payload) {
			body.set('payload', JSON.stringify(payload));
		}
		const res = await fetch(ajaxUrl, {
			method: 'POST',
			credentials: 'same-origin',
			body
		});
		const data = await res.json();
		if (!res.ok) {
			const message = data?.data?.message || data?.message || 'Request failed';
			throw new Error(message);
		}
		if (data && data.success === false) {
			const message = data?.data?.message || data?.message || 'Request failed';
			throw new Error(message);
		}
		return data?.data ?? data;
	};

	const buildLeadMessage = (summary, packet, threadId) => {
		const stamp = new Date().toLocaleString();
		const sender = packet?.email || packet?.username || packet?.user?.email || '';
		const header = [
			'Contact ticket intake',
			`Thread: ${threadId}`,
			`Submitted: ${stamp}`,
			`Client: ${sender || 'Unknown'}`,
			'Tags: contact, sales',
			'AI: pending'
		];
		const body = summary || 'No details provided.';
		return `${header.join('\n')}\n\n${body}`;
	};

	const buildContactThreadId = () => {
		const stamp = new Date();
		const compact = stamp.toISOString().replace(/[-:]/g, '').replace(/\..+/, '');
		const rand = Math.random().toString(36).slice(2, 7);
		return `contact_sales_${compact}_${rand}`;
	};

	const queueAiAssist = async ({ threadId, summary, packet }) => {
		// Placeholder for AI handoff integration.
		// Hook a webhook or AI orchestrator here when ready.
		return { queued: false };
	};

	const sendLeadToMessaging = async (summary) => {
		const packet = getStoredPacket();
		if (!isPacketValid(packet)) return null;
		if (!packet?.access_token) return null;
		const threadId = buildContactThreadId();
		const senderId = packet.email || packet.username || packet.user?.email || '';
		const payload = {
			thread_id: threadId,
			body: buildLeadMessage(summary, packet, threadId),
			sender_id: senderId,
			sender_type: 'client'
		};
		const result = await proxyRequest('messages', 'POST', payload);
		queueAiAssist({ threadId, summary, packet });
		return { threadId, result };
	};

	if (form) {
		form.addEventListener('submit', async (event) => {
			event.preventDefault();
			const summary = buildSummary();
			if (status) status.hidden = true;
			if (submitBtn) submitBtn.hidden = true;
			if (resetBtn) resetBtn.hidden = false;
			form.querySelectorAll('input, select, textarea').forEach((field) => {
				field.disabled = true;
			});
			try { localStorage.setItem('mc_quote_summary', summary); } catch (e) {}
			try {
				await proxyRequest('messages', 'GET', null, { per_page: 1, page: 1 });
				if (status) {
					status.textContent = 'Messaging is connected. Your request is queued for AI intake and a team follow up.';
					status.hidden = false;
				}
				try { localStorage.setItem('mc_quote_pending', '1'); } catch (e) {}
			} catch (err) {
				if (status) {
					status.textContent = `Messaging connection failed: ${err.message || 'Request failed'}. Please call <?php echo esc_js($mc_phone); ?>.`;
					status.hidden = false;
				}
				return;
			}
			// TODO: AI service should create the contact thread and post the intake summary.
		});
	}


	if (resetBtn) {
		resetBtn.addEventListener('click', () => {
			if (form) form.reset();
			if (status) status.hidden = true;
			if (submitBtn) submitBtn.hidden = false;
			if (form) {
				form.querySelectorAll('input, select, textarea').forEach((field) => {
					field.disabled = false;
				});
			}
			updateVehicleFields();
			updateTimeframe();
			if (messagingAvailable === false && submitBtn) {
				submitBtn.disabled = false;
				submitBtn.textContent = callNumber ? 'Call now' : submitLabel;
			} else if (submitBtn) {
				submitBtn.textContent = submitLabel;
			}
		});
	}

	function setMessagingAvailability(online) {
		messagingAvailable = online;
		if (submitBtn) {
			submitBtn.disabled = false;
			submitBtn.textContent = online ? submitLabel : (callNumber ? 'Call now' : submitLabel);
		}
		if (!online && status && callNumber) {
			status.textContent = 'Secure messaging is offline. Please call <?php echo esc_js($mc_phone); ?>.';
			status.hidden = false;
		}
		if (online && status) status.hidden = true;
	}

	async function pingMessaging() {
		if (pingPromise) return pingPromise;
		if (!ajaxUrl) {
			setMessagingAvailability(false);
			return null;
		}
		if (submitBtn) submitBtn.disabled = true;
		pingPromise = proxyRequest('messages', 'GET', null, { per_page: 1, page: 1 })
			.then(() => {
				setMessagingAvailability(true);
			})
			.catch(() => {
				setMessagingAvailability(false);
			})
			.finally(() => {
			pingPromise = null;
		});
		return pingPromise;
	}

})();
</script>

<?php get_footer(); ?>
